<?php

declare(strict_types=1);

namespace Erlage\Photogram\Data\Models\User;

use Erlage\Photogram\System;
use Erlage\Photogram\Data\Query;
use Erlage\Photogram\Data\Common\CommonQueries;
use Erlage\Photogram\Data\Models\Post\PostModel;
use Erlage\Photogram\Data\Tables\Post\PostTable;
use Erlage\Photogram\Data\Tables\User\UserTable;
use Erlage\Photogram\Data\Tables\User\UserBlockTable;
use Erlage\Photogram\Data\Tables\User\UserFollowTable;
use Erlage\Photogram\Data\Tables\Post\PostUserTagTable;
use Erlage\Photogram\Data\Models\User\Block\UserBlockModel;
use Erlage\Photogram\Data\Models\Collection\CollectionModel;
use Erlage\Photogram\Data\Tables\Collection\CollectionTable;
use Erlage\Photogram\Data\Models\User\Follow\UserFollowModel;
use Erlage\Photogram\Data\Models\Post\Usertag\PostUserTagModel;

final class UserModelHelper
{
    public static function updateLastActive(UserModel $userModel): void
    {
        $userModel -> update(array(UserTable::META_LAST_ACTIVE => System::isoDateTime()));

        $userModel -> save();
    }

    public static function updateCacheFollowersCount(UserModel $userModel): void
    {
        $currentCount = (new Query())
            -> from(UserFollowTable::TABLE_NAME)
            -> where(UserFollowTable::FOLLOWED_USER_ID, $userModel -> getId())
            -> count();

        $userModel -> update(array(
            UserTable::CACHE_FOLLOWERS_COUNT => $currentCount,
        ));

        $userModel -> save();
    }

    public static function updateCacheFollowingsCount(UserModel $userModel): void
    {
        $currentCount = (new Query())
            -> from(UserFollowTable::TABLE_NAME)
            -> where(UserFollowTable::FOLLOWED_BY_USER_ID, $userModel -> getId())
            -> count();

        $userModel -> update(array(
            UserTable::CACHE_FOLLOWINGS_COUNT => $currentCount,
        ));

        $userModel -> save();
    }

    public static function updateCachePostsCount(UserModel $userModel): void
    {
        $currentCount = (new Query())
            -> from(PostTable::TABLE_NAME)
            -> where(PostTable::OWNER_USER_ID, $userModel -> getId())
            -> count();

        $userModel -> update(array(
            UserTable::CACHE_POSTS_COUNT => $currentCount,
        ));

        $userModel -> save();
    }

    public static function isUserAvailable(UserModel $targetUserModel, UserModel $currentUserModel): bool
    {
        // if same user
        if ($targetUserModel -> getId() == $currentUserModel -> getId())
        {
            return true;
        }

        return ! ($targetUserModel -> getCacheBlockedUserIds() -> containsValue($currentUserModel -> getId()));
    }

    public static function isUserContentAvailable(UserModel $targetUserModel, UserModel $currentUserModel): bool
    {
        // if same user
        if ($targetUserModel -> getId() == $currentUserModel -> getId())
        {
            return true;
        }

        // if blocked
        if ($targetUserModel -> getCacheBlockedUserIds() -> containsValue($currentUserModel -> getId()))
        {
            return false;
        }

        // if private make sure current user is following
        if ($targetUserModel -> isPrivate())
        {
            return CommonQueries::isFollowing(
                $targetUserModel -> getId(),
                $currentUserModel -> getId()
            );
        }

        return true;
    }

    public static function deleteDependencies(UserModel $userModel): void
    {
        // delete direct child models

        /**
         * @var UserFollowModel[]
         */
        $userFollowModels = CommonQueries::beansWithMatchingPredicates(
            UserFollowTable::getTableName(),
            array(
                UserFollowTable::FOLLOWED_BY_USER_ID => $userModel -> getId(),
            )
        );

        foreach ($userFollowModels as $userFollowModel)
        {
            $userFollowModel -> delete();
        }

        /**
         * @var UserBlockModel[]
         */
        $userBlockModels = CommonQueries::beansWithMatchingPredicates(
            UserBlockTable::getTableName(),
            array(
                UserBlockTable::BLOCKED_BY_USER_ID => $userModel -> getId(),
            )
        );

        foreach ($userBlockModels as $userBlockModel)
        {
            $userBlockModel -> delete();
        }

        // delete side models

        /**
         * @var PostModel[]
         */
        $postModels = CommonQueries::beansWithMatchingPredicates(
            PostTable::getTableName(),
            array(
                PostTable::OWNER_USER_ID => $userModel -> getId(),
            )
        );

        foreach ($postModels as $postModel)
        {
            $postModel -> delete();
        }

        /**
         * @var CollectionModel[]
         */
        $collectionModels = CommonQueries::beansWithMatchingPredicates(
            CollectionTable::getTableName(),
            array(
                CollectionTable::OWNER_USER_ID => $userModel -> getId(),
            )
        );

        foreach ($collectionModels as $collectionModel)
        {
            $collectionModel -> delete();
        }

        /**
         * @var PostUserTagModel[]
         */
        $postUserTagModels = CommonQueries::beansWithMatchingPredicates(
            PostUserTagTable::getTableName(),
            array(
                PostUserTagTable::TAGGED_USER_ID => $userModel -> getId(),
            )
        );

        foreach ($postUserTagModels as $postUserTagModel)
        {
            $postUserTagModel -> delete();
        }
    }
}
